"""
Batch Image Header Watermark Script (High-Contrast, Auto-Fit)
------------------------------------------------------------

Written by ChatGPT at the command of the user.

This script:
- Prompts for watermark text
- Adds a high-contrast header bar to every image
- Auto-fits text size so it never overflows
- Uses outline + shadow so text always stands out
- Processes all images in the same folder
- Saves new watermarked copies (originals untouched)

Requirements:
    pip install pillow

Usage:
    Place this script in a folder with images and run:
        python batch_watermark_header.py
"""

import os
from PIL import Image, ImageDraw, ImageFont

IMAGE_EXTS = (".png", ".jpg", ".jpeg", ".webp", ".bmp", ".tiff")

def load_font(size):
    """Try common fonts; fall back safely."""
    for font_name in ["arialbd.ttf", "arial.ttf"]:
        try:
            return ImageFont.truetype(font_name, size)
        except:
            continue
    return ImageFont.load_default()

def fit_font(draw, text, max_width, start_size):
    """Shrink font until text fits width."""
    size = start_size
    while size > 10:
        font = load_font(size)
        bbox = draw.textbbox((0, 0), text, font=font)
        if bbox[2] - bbox[0] <= max_width:
            return font
        size -= 2
    return load_font(size)

def watermark_image(path, text):
    img = Image.open(path).convert("RGBA")
    w, h = img.size

    header_h = int(h * 0.09)
    header_h = max(header_h, 60)

    new_img = Image.new("RGBA", (w, h + header_h), (0, 0, 0, 255))
    new_img.paste(img, (0, header_h))

    draw = ImageDraw.Draw(new_img)

    # Header bar (semi-transparent black)
    draw.rectangle(
        [(0, 0), (w, header_h)],
        fill=(0, 0, 0, 200)
    )

    # Font sizing
    start_size = int(header_h * 0.6)
    font = fit_font(draw, text, int(w * 0.95), start_size)

    bbox = draw.textbbox((0, 0), text, font=font)
    text_w = bbox[2] - bbox[0]
    text_h = bbox[3] - bbox[1]

    x = (w - text_w) // 2
    y = (header_h - text_h) // 2

    # Shadow
    draw.text((x + 2, y + 2), text, font=font, fill=(0, 0, 0, 255))

    # Main text with outline
    draw.text(
        (x, y),
        text,
        font=font,
        fill=(255, 255, 255, 255),
        stroke_width=2,
        stroke_fill=(0, 0, 0, 255)
    )

    base, ext = os.path.splitext(path)
    out_path = f"{base}_watermarked{ext}"
    new_img.convert("RGB").save(out_path, quality=95)

    print(f"✔ {os.path.basename(out_path)}")

def main():
    text = input("Enter header watermark text: ").strip()

    if not text:
        print("❌ No text entered.")
        return

    images = [f for f in os.listdir(".") if f.lower().endswith(IMAGE_EXTS)]

    if not images:
        print("❌ No images found.")
        return

    print(f"\nProcessing {len(images)} images...\n")

    for img in images:
        try:
            watermark_image(img, text)
        except Exception as e:
            print(f"⚠ Skipped {img}: {e}")

    print("\n✅ Done. Text will always be readable.")

if __name__ == "__main__":
    main()
